/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef VNCRENDERER_H_32178380_c3b4_11e3_9bbd_7071bc8f94f5
#define VNCRENDERER_H_32178380_c3b4_11e3_9bbd_7071bc8f94f5

#include "vncdecoder.h"
#include "vnctypes.h"

/**
 * \file vncrenderer.h
 *
 * This file defines types for use with pluggable renderers provided to
 * the VNC Viewer SDK.
 *
 * For pixel data flow diagrams, please see the documentation for
 * vncdecoder.h.
 */

/**
 * \typedef VNCRendererEventHandle
 *
 * A VNCRendererEventHandle is a platform-specific mechanism for
 * the SDK to detect that there has been activity on a pluggable
 * renderers' context.
 *
 * See VNCRendererContextGetEventHandles() for a full discussion of how
 * the SDK uses VNCRendererContextEventHandle.
 *
 * \see VNCRendererContextGetEventHandles(), VNCRendererContextActivity()
 */
#ifdef _WIN32
    #include <windows.h>
    typedef HANDLE VNCRendererEventHandle;
#else
    typedef int VNCRendererEventHandle;
#endif

/**
 * An enumeration used to describe the direction of activity
 * of interest for each VNCRendererEventHandle.
 *
 * Due to VNCRendererEventHandle being waitable kernel objects on
 * Windows, using any value which is not VNCRendererEventDirectionNone
 * will cause the kernel object to be waited on.
 */
typedef enum
{
  /**
   * Used to indicate that the event handle should not be monitored.
   */
  VNCRendererEventDirectionNone  = 0x0,
  /**
   * Used to indicate that the event handle should only be monitored
   * for read activity.
   */
  VNCRendererEventDirectionRead  = 0x1,
  /**
   * Used to indicate that the event should only be monitored for
   * write activity.
   */
  VNCRendererEventDirectionWrite = 0x2,
  /**
   * Used to indicate that the event should
   * be monitored for both read and write activity.
   */
  VNCRendererEventDirectionBoth  = 0x3
} VNCRendererEventDirection;

/**
 * An enumeration denoting a renderer property. The viewer application
 * can set these properties using vncViewerSetRendererProperty().
 *
 * It is possible to define custom properties, with values starting from
 * VNCRendererPropertyVENDOR.
 */
typedef enum
{
  /**
   * Used to set the visibility of the overlay. pNewValue should point
   * to a vnc_bool_t value, set to true if the overlay should be made
   * visible, and false if the overlay should be hidden.
   */
  VNCRendererPropertyOverlayVisibility = 1,
  /**
   * Used to set the position and size of the overlay. pNewValue should
   * point to a VNCRectangle, which will contain the desired screen
   * coordinates where the overlay should be placed.
   */
  VNCRendererPropertyOverlayRectangle = 2,
  /**
   * Vendors can define custom properties, with keys greater than
   * this value.
   */
  VNCRendererPropertyVENDOR = 0x10000,
  /**
   * Vendors can define custom properties, with keys less than
   * this value. This upper-bound is defined to ensure that undefined
   * behavior is not encountered when casting custom integer values
   * to this enum type.
   */
  VNCRendererPropertyVENDOREnd = 0x10000000,
} VNCRendererProperty;

/**
 * Represents a loaded pluggable renderer.
 *
 * This type is opaque to the SDK, which will pass the value returned
 * by VNCRendererInitialize() to any further renderer interface calls
 * that may require it.
 */
typedef struct VNCRendererFactoryImpl VNCRendererFactory;

/**
 * Represents a context created using a pluggable renderer.
 *
 * This type is opaque to the SDK, which will pass the value returned by
 * VNCRendererContextCreate() to any further renderer interface calls that may
 * require it.
 */
typedef struct VNCRendererImpl VNCRenderer;

/**
 * Identifies the internal data structures of the SDK that correspond to a
 * loaded pluggable renderer factory.
 *
 * This type is opaque to the renderer implementation, which must pass
 * the value supplied to VNCRendererFactoryInitialize() to any
 * supporting API calls that may require it.
 */
typedef void *VNCRendererFactoryContext;

/**
 * Identifies the internal data structures of the SDK that correspond to a
 * loaded pluggable renderer.
 *
 * This type is opaque to the renderer implementation, which must pass the value
 * supplied to VNCRendererCreate() to any supporting API calls that may
 * require it.
 */
typedef void *VNCRendererContext;

/**
 * Identifies the renderer viewer context provided by the user of the viewer SDK.
 *
 * This type is opaque to the SDK, which will just pass the pointer provided to
 * the call to VNCViewerRegisterRenderer() or VNCViewerLoadRendererFromFile().
 */
typedef void *VNCRendererViewerContext;

/**
 * \brief Error codes that may be returned by the implementations of functions
 * in VNCRendererInterface and VNCRendererSupportingAPI.
 *
 * The error codes are split into two ranges:
 *
 *  - 0 to (VNCRendererErrorVENDOR - 1) - these error codes are a mixture of
 *    error codes (e.g.  VNCRendererErrorNoMemory) and
 *    other error conditions that are pre-defined by RealVNC-provided renderers.
 *    You are encouraged to reuse error codes in this range, rather than
 *    defining your own, as long as it makes sense to do so.
 *  - VNCRendererErrorVENDOR and above - this range of error codes is reserved
 *    for the use of third parties developing renderers.  It is intended for
 *    error conditions that are specific to particular renderer implementations,
 *    and that do not map closely to the codes in the 0 to
 *    (VNCRendererErrorVENDOR - 1) range.
 */
typedef enum
{
    /**
     * The operation was successful - no error occurred.
     */
    VNCRendererErrorNone = 0,

    /**
     * No memory was available for the requested operation.
     */
    VNCRendererErrorNoMemory = 1,

    /**
     * The required resources are unavailable.
     */
    VNCRendererErrorResourceUnavailable = 2,

    /**
     * The renderer requires a licensed feature for which the SDK does not have a
     * license.
     */
    VNCRendererErrorFeatureNotLicensed = 3,

    /*
     * The value of one of the parameters to the call is not valid.
     */
    VNCRendererErrorInvalidParameter = 4,

    /*
     * The SDK encountered an unknown internal error.
     */
    VNCRendererErrorSDKInternal = 5,

    /*
     * There was a problem with the provided data.
     */
    VNCRendererErrorMalformedData = 6,

    /*
     * The required resources encountered an error.
     */
    VNCRendererErrorResourceError = 7,

    /*
     * The renderer encountered an unknown internal error.
     */
    VNCRendererErrorInternal = 8,

    /**
     * Start of range of third-party renderer-specific error codes.
     */
    VNCRendererErrorVENDOR = 0x10000
} VNCRendererError;

/**
 * \brief A structure describing the supported features of a pluggable renderer.
 */
typedef struct
{
  /**
   * The human readable name for this implementation of the renderer,
   * encoded as modified UTF-8 with a NUL terminator.
   *
   * This might include the company name or details of the specific
   * hardware requirements.
   *
   * This value may be queried by pluggable decoders, which may use it
   * to determine whether or not they can provide data directly to this
   * renderer using VNCDecoderDrawCustomData, rather than going through
   * the less efficient VNCDecoderDrawRectangle method.
   *
   * This should have been allocated using VNCRendererFactoryAlloc().
   */
  char *pRendererName;
  /**
   * The human readable version for this implementation of the renderer,
   * encoded as modified UTF-8 with a NUL terminator.
   *
   * The format of this string is entirely up to the renderer.
   *
   * This value may be queried by pluggable decoders, which may use it
   * to determine whether or not they can provide data directly to this
   * renderer using VNCDecoderDrawCustomData, rather than going through
   * the less efficient VNCDecoderDrawRectangle method.
   *
   * This should have been allocated using VNCRendererFactoryAlloc().
   */
  char *pRendererVersion;

} VNCRendererSupportInfo;


/**
 * \brief Called by the SDK prior to unloading a renderer DLL or shared object.
 *
 * The implementation should free all resources associated with the
 * renderer implementation.
 *
 * The SDK will call VNCRendererDestroy() for all VNCRenderer objects created by
 * this renderer before calling this method.
 *
 * \param pRendererFactory The renderer factory to be unloaded.
 */
typedef void VNCCALL
VNCRendererFactoryTerminate(VNCRendererFactory *pRendererFactory);

/**
 * \brief Called by the SDK to request that the renderer factory fills in
 * a structure with support information.
 *
 * Ownership of any strings contained in pRendererSupportInfo is transferred
 * to the SDK if VNCRendererErrorNone is returned.
 *
 * \param pRendererFactory The renderer factory to request information about.
 * \param pRendererSupportInfo On success this should have been filled
 * with information describing the renderer.
 * \param rendererSupportInfoSize The size of the memory pointed to by
 * pRendererSupportInfo.
 *
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererFactoryGetSupportInfo(VNCRendererFactory *pRendererFactory,
                                VNCRendererSupportInfo *pRendererSupportInfo,
                                size_t rendererSupportInfoSize);

/**
 * \brief Called by the SDK at the start of a connection to create
 * a per-connection instance of a renderer.
 *
 * Failure to create a renderer will not be treated by the SDK as a
 * fatal error, and viewer applications should revert to rendering
 * the contents of the framebuffer directly when a renderer is not
 * loaded. Applications can tell whether or not a renderer has failed
 * to load using the VNCViewerPluggableRendererLoadCallback callback.
 *
 * \param pRendererFactory The renderer factory to use.
 * \param rendererContext Opaque data type to be provided to applicable
 * supporting APIs.
 *
 * \return A pointer for use in other renderer calls by the SDK, or
 * NULL on failure.
 *
 * \see VNCRendererDestroy
 */
typedef VNCRenderer *VNCCALL
VNCRendererCreate(VNCRendererFactory *pRendererFactory,
                 VNCRendererContext rendererContext);

/**
 * \brief Called by the SDK at the end of a connection to destroy
 * a previously created renderer.
 *
 * This will be called by the SDK as part of a connection being
 * destroyed. The pointer provided will match one returned from a
 * previous call to VNCRendererCreate.
 *
 * Renderers should remove any created overlays and free any system
 * resources in this method.
 *
 * \param pRenderer The renderer to destroy.
 */
typedef void VNCCALL
VNCRendererDestroy(VNCRenderer *pRenderer);

/**
 * \brief Called by the SDK to retrieve the current
 * VNCRendererEventHandle objects for the given renderer.
 *
 * A renderer can provide zero or more handles for the SDK to
 * monitor for it. The SDK obtains these event handles by calling this
 * method on every iteration of its main loop. When activity is detected
 * on these handles the SDK will call VNCRendererActivity(). This prevents
 * the renderer from needing to run a separate thread for performing such
 * asynchronous actions.
 *
 * On Windows, VNCRendererEventHandle is a HANDLE that must identify
 * a waitable kernel object. On UNIX platforms,
 * VNCRendererEventHandle is a file descriptor suitable for use in a
 * select() call.
 *
 * On successful return, *ppEventHandles should be set to point to an
 * array of handles and *ppEventDirections should point to an equal
 * sized array of VNCRendererEventDirection values. These arrays should
 * remain valid until the next call to VNCRendererGetEventHandles() or
 * VNCRendererDestroy() for the renderer.
 *
 * \param pRenderer The renderer to use.
 * \param ppEventHandles An out parameter pointing to an array of event handles.
 * \param ppEventDirection An out parameter pointing to an array indicating the
 * direction of interest for the handles in ppEventHandles.
 * \param eventHandleCount The number of event handles in the arrays pointed to
 * by *ppEventHandles and *ppEventDirection.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererGetEventHandles(VNCRenderer *pRenderer,
                          VNCRendererEventHandle **ppEventHandles,
                          VNCRendererEventDirection **ppEventDirections,
                          size_t *eventHandleCount);

/**
 * \brief Called by the SDK when activity is detected on an event handle.
 *
 * When activity is detected in the appropriate direction on one or
 * more of the event handles provided by the previous call to
 * VNCRendererGetEventHandles() then the SDK will call this method.
 *
 * The renderer should take care to clear any pending activity on the
 * handles or to not include them in the next call to
 * VNCRendererGetEventHandles(), otherwise the SDK will call this method
 * almost immediately again.
 *
 * \param pRenderer The renderer to use.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererActivity(VNCRenderer *pRenderer);

/**
 * \brief Called by the SDK before the first rectangle is received, to
 * notify the renderer of the current server pixel format. It is called again
 * whenever the pixel format changes.
 *
 * The pixel format in use over the RFB protocol can change during the course
 * of a connection. This method is called by the SDK so that it can take any
 * changes in pixel format into account when rendering data.
 *
 * pNewPixelFormat is only valid for the duration of this method
 * call, so it is recommended that the renderer makes a copy of it
 * before returning.
 *
 * \param pRenderer The renderer to use.
 * \param pNewPixelFormat The new pixel format to use.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererPixelFormatChange(VNCRenderer *pRenderer,
                             const VNCPixelFormat *pNewPixelFormat);

/**
 * \brief Called by the SDK to notify the renderer whenever a
 * SetColourMapEntries message is received from the server.
 *
 * The color map can change during the course of a connection. This
 * method is called by the SDK so that the renderer can take any changes
 * into account when rendering data.
 *
 * Since the viewer has control over the pixel format in use, it is
 * recommended that a true-color pixel format is chosen, as this will
 * lead to best performance. Indexed color formats may be desirable
 * over low-bandwidth connections, but will have a negative impact
 * on high-handwidth connections (like USB). When a true-color pixel
 * format is in use, the colour map can be ignored.
 *
 * pColorMap is only valid for the duration of this method
 * call, so it is recommended that the renderer makes a copy of it
 * before returning.
 *
 * \param pRenderer The renderer to use.
 * \param firstColor The index of the first color in the provided map.
 * \param numberOfEntries The number of entries in the color map.
 * \param pColorMap The color map. This is an array of 3 * numberOfEntries
 * 16-bit unsigned integers, representing red, green, and blue intensities
 * for each of the colors in the map. See the RFB protocol specification
 * for more detail.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererColorMapChange(VNCRenderer *pRenderer,
                          vnc_uint16_t firstColor,
                          vnc_uint16_t numberOfEntries,
                          const vnc_uint16_t *pColorMap);

/**
 * \brief Called by the SDK before the first rectangle is received, to
 * notify the renderer of the current remote display size. It is called again
 * whenever the display size changes.
 *
 * Any operations needed to accomodate the new remote display size should
 * be performed in response to receiving this method call.
 *
 * \param pRenderer The renderer to use.
 * \param width The new width of the server display.
 * \param height The new height of the server display.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererDesktopResize(VNCRenderer *pRenderer,
                        vnc_uint16_t width,
                        vnc_uint16_t height);

/**
 * \brief Called by the SDK when the viewer application sets a renderer
 * property, using VNCViewerSetRendererProperty().
 *
 * Any operations needed to accomodate the new value should be performed
 * in response to receiving this method call.
 *
 * \param pRenderer The renderer to use.
 * \param propertyKey The key representing the property being set.
 * \param newValue A pointer to the new value of the property. This is owned
 * by the viewer application, and will be valid for the duration of this
 * method call.
 * \param valueSize The size of the value pointed to by newValue, in bytes.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererPropertySet(VNCRenderer *pRenderer,
                       VNCRendererProperty propertyKey,
                       const void* newValue,
                       size_t valueSize);

/**
 * \brief Called by the SDK when a framebuffer update has started. The
 * renderer should expect to start receiving rectangles, which should
 * only be flushed to the screen when VNCRendererOnFramebufferUpdateEnd
 * is called.
 *
 * \param pRenderer The renderer to use.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererOnFramebufferUpdateStart(VNCRenderer *pRenderer);

/**
 * \brief Called by the SDK when a rectangle is decoded by an internal
 * decoder, or a pluggable decoder renders a rectangle using
 * VNCDecoderDrawRectangle().
 *
 * The data provided is represented using the pixel format specified by
 * pFormat. The renderer should render this data to its overlay, performing
 * a pixel format conversion if necessary.
 *
 * The parameters provided to this method are owned by the viewer SDK,
 * and are guaranteed to be valid only for the duration of this method call.
 *
 * \param pRenderer The renderer to use.
 * \param pRect The size and position of the rectangle to be drawn.
 * \param pFormat The pixel format of the rectangle to be drawn.
 * \param pBuf A pointer to the buffer containing the pixel data.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererDrawRGBRectangle(VNCRenderer *pRenderer,
                            const VNCRectangle *pRect,
                            const VNCPixelFormat *pFormat,
                            const vnc_uint8_t *pBuf);

/**
 * \brief Called by the SDK when the renderer should copy a rectangle from
 * one part of the overlay to another. This usually occurs in response to a
 * rectangle received using the CopyRect encoding.
 *
 * The parameters provided to this method are owned by the viewer SDK,
 * and are guaranteed to be valid only for the duration of this method call.
 *
 * \param pRenderer The renderer to use.
 * \param pRectFrom The size and position of the rectangle to be copied.
 * \param pRectTo The position to which the rectangle should be copied. The
 * width and height of this rectangle should be identical to that of
 * pRectFrom.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererCopyRectangle(VNCRenderer *pRenderer,
                         const VNCRectangle *pRectFrom,
                         const VNCRectangle *pRectTo);

/**
 * \brief Called by the SDK when the renderer should fill in a rectangle
 * of pixels with a particular colour.
 *
 * The parameters provided to this method are owned by the viewer SDK,
 * and are guaranteed to be valid only for the duration of this method call.
 *
 * \param pRenderer The renderer to use.
 * \param pRect The size and position of the rectangle to fill in.
 * \param pFormat The pixel format of the provided pixel.
 * \param pBuf A pointer to a buffer containing one pixel of data, encoded
 * in the format specified by pFormat.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererFillRectangle(VNCRenderer *pRenderer,
                         const VNCRectangle *pRect,
                         const VNCPixelFormat *pFormat,
                         const vnc_uint8_t *pBuf);

/**
 * \brief Called by the SDK when the renderer should render data generated
 * by a pluggable decoder.
 *
 * The data provided should be in a structure agreed on in advance by the
 * decoder and the renderer. The type of this data is opaque to the
 * SDK.
 *
 * The parameters provided to this method are guaranteed to be valid only
 * for the duration of this method call.
 *
 * \param pRenderer The renderer to use.
 * \param pRect The size and position of the rectangle to be drawn.
 * \param pData A pointer to the data provided by the pluggable decoder.
 * \param dataSize The size of the value pointed to by pData, in bytes.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererDrawCustomData(VNCRenderer *pRenderer,
                          const VNCRectangle *pRect,
                          const void* pData,
                          size_t dataSize);

/**
 * \brief Called by the SDK when a framebuffer update has been completed.
 * The renderer should flush all rectangles received to the screen at this
 * point.
 *
 * \param pRenderer The renderer to use.
 * \return VNCRendererErrorNone on success, or some other VNCRendererError on
 * failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererOnFramebufferUpdateEnd(VNCRenderer *pRenderer);


/**
 * \brief APIs to be implemented by the renderer and called by the SDK.
 */
typedef struct {
  /**
   * Called by the SDK prior to unloading a renderer factory DLL or
   * shared object.
   */
  VNCRendererFactoryTerminate *vncRendererFactoryTerminate;
  /**
   * Called by the SDK to retrieve information about the renderer
   * factory support.
   */
  VNCRendererFactoryGetSupportInfo *vncRendererFactoryGetSupportInfo;
  /**
   * Called by the SDK to create a renderer for a connection.
   */
  VNCRendererCreate *vncRendererCreate;
  /**
   * Called by the SDK to destroy a renderer for a connection.
   */
  VNCRendererDestroy *vncRendererDestroy;
  /**
   * Called by the SDK to retrieve a list of VNCRendererEventHandle values for
   * the provided renderer.
   */
  VNCRendererGetEventHandles *vncRendererGetEventHandles;
  /**
   * Called by the SDK to indicate that there is activity on the event handles.
   */
  VNCRendererActivity *vncRendererActivity;
  /**
   * Called by the SDK to notify the renderer of a change to the
   * wire pixel format.
   */
  VNCRendererPixelFormatChange *vncRendererPixelFormatChange;
  /**
   * Called by the SDK to notify the renderer of a change to the
   * color map.
   */
  VNCRendererColorMapChange *vncRendererColorMapChange;
  /**
   * Called by the SDK to notify the renderer of a change to the
   * display size.
   */
  VNCRendererDesktopResize *vncRendererDesktopResize;
  /**
   * Called by the SDK to indicate that the viewer application has set a
   * renderer property.
   */
  VNCRendererPropertySet *vncRendererPropertySet;
  /**
   * Called by the SDK to indicate that a framebuffer update has started.
   */
  VNCRendererOnFramebufferUpdateStart *vncRendererOnFramebufferUpdateStart;
  /**
   * Called by the SDK to indicate that an RGB rectangle should be drawn.
   */
  VNCRendererDrawRGBRectangle *vncRendererDrawRGBRectangle;
  /**
   * Called by the SDK to indicate that a rectangle should be copied from
   * one part of the overlay to another.
   */
  VNCRendererCopyRectangle *vncRendererCopyRectangle;
  /**
   * Called by the SDK to indicate that a rectangle should be filled with
   * a particular colour of pixels.
   */
  VNCRendererFillRectangle *vncRendererFillRectangle;
  /**
   * Called by the SDK when arbitrary data is provided by the decoder.
   */
  VNCRendererDrawCustomData *vncRendererDrawCustomData;
  /**
   * Called by the SDK to indicate that a framebuffer update has ended,
   * and the rectangles provided should be drawn to the screen.
   */
  VNCRendererOnFramebufferUpdateEnd *vncRendererOnFramebufferUpdateEnd;
} VNCRendererInterface;


/**
 * \brief Called by the renderer factory to write to the SDK's log.
 *
 * VNCRendererLog() is preferred to VNCRendererFactoryLog() because
 * VNCRendererLog() is guaranteed to work correctly if called from any
 * thread. VNCRendererFactoryLog() does not work if called from a
 * thread that was started by the renderer.
 *
 * Otherwise, this method is identical to VNCRendererLog().
 *
 * \param rendererFactoryContext The renderer VNCRendererFactoryContext
 * that was passed to VNCRendererFactoryInitialize().
 * \param text The text to write to the log, as a NUL-terminated UTF-8
 * string.
 */
typedef void VNCCALL
VNCRendererFactoryLog(VNCRendererFactoryContext rendererFactoryContext,
                     const char *text);

/**
 * \brief Called by the renderer to allocate memory whose ownership is to be
 * transferred to the SDK.
 *
 * The start address of the returned memory is aligned to a word boundary, and
 * the memory is filled with zeroes.
 *
 * If ownership has been passed to the SDK, then the SDK will free the allocated
 * memory when it is no longer needed. If ownership is not transferred, then
 * the plugin should free the memory using VNCRendererFactoryFree().
 *
 * \param rendererFactoryContext The VNCRendererFactoryContext that was passed to
 * VNCRendererFactoryInitialize().
 * \param size The size of data to be allocated.
 *
 * \return A pointer to the allocated memory or NULL on failure.
 */
typedef void *VNCCALL
VNCRendererFactoryAlloc(VNCRendererFactoryContext rendererFactoryContext,
                       size_t size);

/**
 * \brief Called by the renderer to allocate memory whose ownership is to be
 * transferred to the SDK. The contents of the allocated memory will be equal
 * to the supplied string.
 *
 * If ownership has been passed to the SDK, then the SDK will free the allocated
 * memory when it is no longer needed. If ownership is not transferred, then
 * the plugin should free the memory using VNCRendererFactoryFree().
 *
 * \param rendererFactoryContext The VNCRendererFactoryContext that was passed to
 * VNCRendererFactoryInitialize().
 * \param str The NUL-terminated UTF-8 string to copy into the allocated memory.
 *
 * \return A pointer to the allocated memory or NULL on failure.
 */
typedef char *VNCCALL
VNCRendererFactoryAllocString(VNCRendererFactoryContext rendererFactoryContext,
                             const char* str);

/**
 * \brief Called by the renderer to free memory which was allocated by
 * VNCRendererFactoryAlloc or VNCRendererFactoryAllocString, but whose
 * ownership was not transferred to the SDK.
 *
 * \param rendererFactoryContext The VNCRendererFactoryContext that was passed to
 * VNCRendererFactoryInitialize().
 * \param buffer The buffer to be freed.
 */
typedef void VNCCALL
VNCRendererFactoryFree(VNCRendererFactoryContext rendererFactoryContext,
                      void *buffer);

/**
 * \brief Called by the renderer to write to the SDK's log.
 *
 * Renderer implementations are encouraged to log significant events during
 * renderer creation and activation, and any other information that may help offline
 * diagnosis of problems.
 *
 * It is particularly important to log the reason for a failure if
 * the reason cannot easily be conveyed with an error code.
 *
 * VNCRendererLog() is preferred to VNCRendererFactoryLog() because
 * VNCRendererLog() is guaranteed to work correctly if called from any
 * thread. VNCRendererFactoryLog() does not work if called from a
 * thread that was started by the renderer.
 *
 * \param rendererContext The VNCRendererContext that was passed to
 * VNCRendererCreate().
 * \param text The text to write to the log.
 */
typedef void VNCCALL
VNCRendererLog(VNCRendererContext rendererContext, const char *text);

/**
 * \brief Called by the renderer to check whether a particular feature is
 * licensed.
 *
 * This method allows you to ensure that your custom renderers are usable only in
 * certain applications.  The feature check succeeds if the SDK is authorized
 * to use at least one of the requested features.  The features are searched
 * for both in the the licenses available to the SDK and in the set of features
 * defined by calls by the application to the SDK's API.
 *
 * If none of the features are licensed, then the renderer may either continue to
 * operate, restricting functionality as appropriate, or return
 * VNCRendererErrorFeatureNotLicensed to the SDK, in which case the SDK will
 * terminate the session.
 *
 * \param rendererContext The VNCRendererContext that was passed to
 * VNCRendererCreate().
 * \param featureID The feature identifier for which to check for a license.
 * \param *pResult On successful return, *pResult is non-zero if any of the
 * features is licensed, and zero if not.
 * \retval VNCRendererErrorNone The feature check was made successfully.  This
 * does *not* imply that any of the features is licensed; check *pResult.
 * \retval VNCRendererErrorInvalidParameter featureIDs or pResult is NULL, or
 * featureIDCount is zero.
 *
 * pResult is non-zero if the feature is licensed, or zero if not.
 */
typedef VNCRendererError VNCCALL
VNCRendererLocalFeatureCheck(VNCRendererContext rendererContext,
                            const unsigned *featureIDs,
                            size_t featureIDCount,
                            int *pResult);

/**
 * \brief Called by the renderer to notify the SDK that a framebuffer
 * update has been drawn. This will be taken into account in the SDK's
 * performance measurements.
 *
 * \note If the renderer does not call this support API, then the SDK will
 * consistently report a value of zero for its FramesRendered performance
 * metric (when enabled). Pluggable renderers which do not use this API will
 * need to monitor and report frame rate information themselves, if this is
 * required. Pluggable renderers can write performance information to the SDK's
 * log.
 *
 * \param rendererContext The VNCRendererContext that was passed to
 * VNCRendererCreate().
 *
 * \see VNCParameterPerformanceMeasuring
 */
typedef void VNCCALL
VNCRendererOnFrameRendered(VNCRendererContext rendererContext);

/**
 * \brief Called by the renderer to instruct the Viewer SDK to draw
 * the specified pixel data to its framebuffer.
 *
 * The data provided is represented using the pixel format specified 
 * by "format".
 *
 * The parameters provided to this method are owned by the renderer,
 * and must be valid for the duration of this method call.
 *
 * \param rendererContext The VNCRendererContext that was passed to
 * VNCRendererCreate().
 * \param rect The size and position of the rectangle to be drawn.
 * \param format The pixel format of the rectangle to be drawn.
 * \param buf A pointer to the buffer containing the pixel data.
 * \return VNCRendererErrorNone on success, or some other
 * VNCRendererError on failure.
*/
typedef VNCRendererError VNCCALL
VNCRendererSupportingAPIDrawRGBRectangle(
        VNCRendererContext rendererContext,
        const VNCRectangle *rect,
        const VNCPixelFormat *format,
        const vnc_uint8_t *buf);


/**
 * \brief Called by the renderer to instruct the Viewer SDK to copy
 * the specified rectangle from one location to another in its framebuffer.
 *
 * The parameters provided to this method are owned by the renderer,
 * and must be valid for the duration of this method call.
 *
 * \param renderer The renderer to use.
 * \param rectFrom The size and position of the rectangle to be copied.
 * \param rectTo The position to which the rectangle should be copied. The
 * width and height of this rectangle should be identical to that of
 * rectFrom.
 * \return VNCRendererErrorNone on success, or some other
 * VNCRendererError on failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererSupportingAPICopyRectangle(
        VNCRendererContext rendererContext,
        const VNCRectangle *rectFrom,
        const VNCRectangle *rectTo);


/**
 * \brief Called by the renderer to instruct the Viewer SDK to fill
 * the specified rectangle with the specified color.
 *
 * The parameters provided to this method are owned by the renderer,
 * and must be valid for the duration of this method call.
 *
 * \param rendererContext The VNCRendererContext that was passed to
 * VNCRendererCreate().
 * \param rect The size and position of the rectangle to fill in.
 * \param format The pixel format of the provided pixel.
 * \param buf A pointer to a buffer containing one pixel of data, encoded
 * in the format specified by format.
 * \return VNCRendererErrorNone on success, or some other
 * VNCRendererError on failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererSupportingAPIFillRectangle(
        VNCRendererContext rendererContext,
        const VNCRectangle *rect,
        const VNCPixelFormat *format,
        const vnc_uint8_t *buf);

/**
 * \brief Called by the renderer to instruct the Viewer SDK to 
 * request a non-incremental framebuffer update.
 *
 * Once the call returns, the pluggable renderer may still receive
 * at most one incremental framebuffer update before receiving the
 * non-incremental framebuffer update.
 *
 * If the call is invoked when the framebuffer update is disabled
 * (see VNCViewerEnableDisplayUpdates), then the non-incremental
 * framebuffer update request will be delayed until the framebuffer
 * update is enabled.
 *
 * \param rendererContext The VNCRendererContext that was passed to
 * VNCRendererCreate().
 * \return VNCRendererErrorNone on success, or some other
 * VNCRendererError on failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererSupportingAPIRequestNonIncrementalFramebufferUpdate(
        VNCRendererContext rendererContext);


/**
 * \brief Called by the renderer to inform the viewer application whether
 * or not it is necessary to render the framebuffer.
 *
 * For example, when the renderer is displaying an overlay, there is no need
 * for the viewer application to render the framebuffer itself.
 *
 * If shouldRenderFramebuffer is set to vnc_true, then the viewer application
 * must render the framebuffer. Otherwise, it can stop rendering the
 * framebuffer.
 *
 * \param rendererContext The VNCRendererContext that was passed to
 * VNCRendererCreate().
 * \param shouldRenderFramebuffer vnc_true if the viewer application must
 * render the framebuffer. Otherwise, it should be vnc_false.
 * \return VNCRendererErrorNone on success, or some other
 * VNCRendererError on failure.
 */
typedef VNCRendererError VNCCALL
VNCRendererSupportingAPINotifyFramebufferState(
		VNCRendererContext rendererContext,
		vnc_bool_t shouldRenderFramebuffer);

/**
 * \brief Called by the renderer to instruct the Viewer SDK to register a
 * pluggable decoder.
 *
 * A pluggable renderer may also act as a pluggable decoder, in which case it
 * must call this API to additionally register itself as a decoder with the
 * SDK. The pluggable renderer may only call this API from its implementation
 * of VNCRendererFactoryInitialize(). A decoder factory must be provided, to
 * allow the SDK to generate decoders as required. The SDK guarantees to
 * destroy decoder factories before destroying renderer factories, meaning that
 * a pluggable renderer can safely provide a decoder factory.
 *
 * See \ref VNCViewerRegisterDecoder for more information on registering
 * pluggable decoders with the Viewer SDK. See \ref vncdecoder.h for more
 * information on implementing a pluggable decoder.
 *
 * \param rendererFactoryContext The \ref VNCRendererFactoryContext that was
 * passed to VNCRendererFactoryInitialize().
 * \param decoderInitialize A pointer to the decoder factory initialization
 * function.
 * \param decoderViewerContext A pointer to a location of the caller's
 * choosing, which will be passed unaltered to the
 * \ref VNCDecoderFactoryInitialize function.
 *
 * \return VNCViewerErrorNone The pluggable decoder was registered
 * successfully.
 * \return VNCViewerErrorInvalidParameter The pluggable decoder could not be
 * registered, as either rendererFactoryContext or decoderInitialize are NULL.
 * \return VNCViewerErrorFeatureNotLicensed The pluggable decoder could not be
 * registered, as the Viewer SDK is not licensed to use pluggable decoders.
 * \return VNCViewerErrorDecoderLoadFailed The pluggable decoder could not be
 * registered, as either the decoder factory could not be initialized or this
 * API was called outside of VNCRendererFactoryInitialize().
 * \return VNCViewerErrorDecoderDuplicateEncoding The pluggable decoder could
 * not be registered, as the encoding it provides is already handled by a
 * previously loaded pluggable decoder.
 * \return VNCViewerErrorDecoderUnsupportedEncoding The pluggable decoder could
 * not be registered, as the encoding it provides is not supported by the
 * Viewer SDK.
 *
 * \see VNCViewerRegisterDecoder, vncdecoder.h
 */
typedef VNCViewerError VNCCALL
VNCRendererSupportingAPIRegisterDecoder(
    VNCRendererFactoryContext rendererFactoryContext,
    VNCDecoderFactoryInitializeType *decoderInitialize,
    VNCDecoderViewerContext decoderViewerContext);

/**
 * \brief APIs provided by the SDK for calling by the renderer.
 */
typedef struct {
  /**
   * Called by the renderer factory to write to the SDK's log.
   */
  VNCRendererFactoryLog *vncRendererFactoryLog;
  /**
   * Called by the renderer to allocate memory whose ownership is to be
   * transferred to the SDK.
   */
  VNCRendererFactoryAlloc *vncRendererFactoryAlloc;
  /**
   * Called by the renderer to allocate memory whose ownership is to be
   * transferred to the SDK.
   */
  VNCRendererFactoryAllocString *vncRendererFactoryAllocString;
  /**
   * Called by the renderer to free memory whose ownership has not been
   * transferred to the SDK.
   */
  VNCRendererFactoryFree *vncRendererFactoryFree;
  /**
   * Called by the renderer to write to the SDK's log.
   */
  VNCRendererLog *vncRendererLog;
  /**
   * Called by the renderer to check whether a particular features is licensed.
   */
  VNCRendererLocalFeatureCheck *vncRendererLocalFeatureCheck;
  /**
   * Called by the renderer to notify the SDK that a frame has been rendered.
   */
  VNCRendererOnFrameRendered *vncRendererOnFrameRendered;

  /**
   * Called by the renderer to instruct the Viewer SDK to draw the specified
   * pixel data to its framebuffer.
   */
  VNCRendererSupportingAPIDrawRGBRectangle *vncRendererSupportingAPIDrawRGBRectangle;

  /**
   * Called by the renderer to instruct the Viewer SDK to copy the specified
   * rectangle from one location to another in its framebuffer.
   */
  VNCRendererSupportingAPICopyRectangle *vncRendererSupportingAPICopyRectangle;

  /**
   * Called by the renderer to instruct the Viewer SDK to fill the specified
   * rectangle with the specified color.
   */
  VNCRendererSupportingAPIFillRectangle *vncRendererSupportingAPIFillRectangle;

  /**
   * Called by the renderer to instruct the Viewer SDK to request a
   * non-incremental framebuffer update.
   **/
  VNCRendererSupportingAPIRequestNonIncrementalFramebufferUpdate
          *vncRendererSupportingAPIRequestNonIncrementalFramebufferUpdate;

  /**
   * Called by the renderer to inform the viewer application whether
   * or not it should render the framebuffer.
   */
  VNCRendererSupportingAPINotifyFramebufferState
          *vncRendererSupportingAPINotifyFramebufferState;

  /**
   * Called by the renderer to instruct the Viewer SDK to register a pluggable
   * decoder.
   */
  VNCRendererSupportingAPIRegisterDecoder *vncRendererSupportingAPIRegisterDecoder;

} VNCRendererSupportingAPI;

/**
 * \brief The type of the entry point to be exported by a renderer DLL
 * or shared object.
 *
 * \param rendererFactoryContext Opaque data type to be provided to
 * applicable supporting APIs.
 * \param rendererViewerContext Opaque data type as provided by the
 * viewer application to the SDK when the renderer was registered or
 * loaded.
 * \param pRendererInterface Structure to contain the addresses of the
 * renderers' implementation functions.  The implementation of
 * VNCRendererFactoryInitialize() must fill in this structure before
 * returning.
 * \param rendererInterfaceSize The size of *pRendererInterface.
 * \param pRendererSupportingAPI Structure containing functions provided by the
 * SDK for calling by the renderer.
 * \param rendererSupportingAPISize The size of *pRendererSupportingAPI.
 *
 * \return The new renderer object.  The SDK will call VNCRendererTerminate() when
 * this object is no longer required. Returning NULL here indicates that an
 * error has occurred, and the renderer factory will not be initialised.
 */
typedef VNCRendererFactory *VNCCALL VNCRendererFactoryInitializeType(
    VNCRendererFactoryContext rendererFactoryContext,
    VNCRendererViewerContext rendererViewerContext,
    VNCRendererInterface *pRendererInterface,
    size_t rendererInterfaceSize,
    const VNCRendererSupportingAPI *pRendererSupportingAPI,
    size_t rendererSupportingAPISize);

#endif // VNCRENDERER_H_32178380_c3b4_11e3_9bbd_7071bc8f94f5

